<?php

namespace App\Http\Controllers\Admin\Role;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;

class RoleController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $roles = Role::withCount('permissions')->where('id', '!=', 1)->orderBy('name')->get();

        return view('admin.role.index', compact('roles'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $role = new Role;

        return view('admin.role.create', compact('role'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => 'required|string|max:255|unique:roles',
        ]);
        $data['status'] = $request->has('status');

        try {
            $role = Role::updateOrCreate([
                'name' => $data['name'],
            ], $data);

            return redirect()->route('admin.role.index')->with('success', 'Role created successfully.');
        } catch (\Exception $e) {
            // return $e->getMessage();
            return redirect()->back()->with('error', 'Role could not be created.');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Role $role)
    {
        abort(403);

        return view('admin.role.show', compact('role'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Role $role)
    {
        // return ucwords(str_replace('-', ' ', 'access-dashboard'));
        $permissions = Permission::select('id', 'name', 'module')->get()->groupBy('module');

        return view('admin.role.permission', compact('role', 'permissions'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Role $role)
    {
        $request->validate([
            'permissions' => 'required|array',
            'permissions.*' => 'required|exists:permissions,name',
        ]);

        try {
            $role->syncPermissions($request->permissions);
            app()[\Spatie\Permission\PermissionRegistrar::class]->forgetCachedPermissions();

            return redirect()->route('admin.role.index')->with('success', 'Role permissions updated successfully.');
        } catch (\Exception $e) {
            return $e->getMessage();

            return redirect()->back()->with('error', 'Role could not be updated.');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Role $role)
    {
        try {
            if ($role->removable) {
                $role->delete();
                $role->permissions()->detach();
            }

            return response()->json(['message' => 'Role deleted successfully.', 'status' => 'success']);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Role could not be deleted.', 'status' => 'error'], 500);
        }
    }
}
